﻿//
// 3D物理エンジン Open Dynamics Engine を用いた1輪車のプログラム
// Open Dynamics Engine 本家: http://www.ode.org/
// Open Dynamics Engine Wiki: http://opende.sourceforge.net/wiki/index.php/Main_Page
// 参考サイト1: http://www.crystal-creation.com/
// 参考サイト2: http://demura.net/
// 参考サイト3: http://www.koj-m.sakura.ne.jp/ode/index.php?Open%20Dynamics%20Engine
// 参考サイト4: http://hwm7.gyao.ne.jp/shinpuku/index.html
// 参考サイト5: http://www10.atwiki.jp/bambooflow/
//

//
// GNU Scientific Library
//
// 本家: http://www.gnu.org/software/gsl/
// 参考サイト1: http://www.tmps.org/index.php?TMPSwiki
// 参考サイト2: http://gonzaburou.cocolog-nifty.com/blog/2006/11/gslvisual_cc_8716.html
//

#ifndef MAIN_H // 二重インクルードを防止
#define MAIN_H

//
// for Open Dynamics Engine
//
#include <ode/ode.h>             // ODEを使うことを開発環境にお知らせ
#include <drawstuff/drawstuff.h> // 画面表示を受け持つ部分

#define DEBUG 1     // 1:デバッグを有効 0:デバッグを無効
#define SAMPLINGTIME (50.0) // [msec]で制御対象の状態をサンプリング
#define GRAVITY (-9.8) // -9.8[m/{sec}^2]
#define MAX_CONTACTS (20) // 衝突検出の精度: 大きいほど精度が上がるが、計算コストがかかる
#define DT  ((double)((double)(SAMPLINGTIME) / (1000.0))) // ODEの世界での物理演算の刻み時間 dt[sec] ***この値を小さくするほど精度は上がるが計算時間がかかる***
//#define FPS 60.0 // [frame/sec] 1秒間に何回画面を更新するか
//#define SPF (1.0 / FPS) // [sec/frame] 1フレームに何秒要するか

//
// for GNU Scientific Library
//
#include "Vector.h"                  // ベクトルを扱うC++でラッパークラス
#include "Matrix.h"                  // 行列を扱うC++でラッパークラス
#pragma comment(lib, "gsl.lib")      // GSLを使用するためのスタティックリンクライブラリ
#pragma comment(lib, "gslcblas.lib") // GSLを使用するためのスタティックリンクライブラリ

//
// Open Dynamics Engineを使用する時、Windows上で精度を補償するためのおまじない
//
#ifdef _MSC_VER
#pragma warning(disable:4244 4305)  // for VC++, no precision loss complaints
#endif


#include <stdio.h>   /* c言語 標準入出力 */
#include <stdlib.h>  /* c言語 基本関数を含む */
#include <string.h>  /* c言語 文字列操作ライブラリ */
#include <math.h>    /* c言語 数学ライブラリ */
#include <time.h>    /* c言語 時間の扱い */

#include <iostream> // c++ 標準入出力
#include <iomanip>  // c++ 標準入出力マニピュレータ
#include <fstream>  // c++ ファイルハンドリン

//
// Open Dynamics Engine の内部仕様の古い関数を新しい関数で暗黙的に置き換え
//
#ifdef dDOUBLE
#define dsDrawSphere  dsDrawSphereD
#define dsDrawBox     dsDrawBoxD
#define dsDrawCylinder dsDrawCylinderD
#define dsDrawCapsule  dsDrawCapsuleD
#define dsDrawLine     dsDrawLineD
#define dsDrawTriangle dsDrawTriangleD
#endif

//
// 物体の情報を格納する構造体
//
typedef struct {
  dBodyID body;                        // ボディのID番号(描画と物体の存在を格納)
  dGeomID geom;                        // ジオメトリのID番号(衝突計算に使う幾何情報)
} MyObject;                            // MyObject構造体

enum
{
	LearningControl = 1, // 学習制御
	InitialValueResponse, // 初期値応答
	AdaptiveControl, // 適応制御
};

//
// 新しい変数
//
int ControlFlag; // 学習制御、初期値応答、適応制御のどれか
int LearningLength; // 現在の学習回数、クロック
int LearningLengthFinal; // 最終学習回数
int Dimension; // 使用する状態変数の次数 (例)[x0, x1, x2] --> 3次 旧 DD
int NumberOfInput; // 使用する入力数
int BlockLength; // 現在のブロック長さ、多段決定に用いられる区間
int BlockLengthFinal; // ブロック長の終端長さ
int RunTime; // [msec] 実行時間

double Sigma; // 追加入力雑音の振幅 σ
double Delta; // 逆行列補題の初期値 δ 小さいほど理論に忠実となるが計算機の扱える桁に注意する
double Lambda1; // λ1 [0<=1] ゲイン更新に関わる学習係数
double Lambda2; // λ2 [0<=1] ゲイン更新に関わる入力係数

CMatrix x; // 状態変数 x(=sensor value)
CMatrix z; // 応答信号 z
CMatrix u; // 制御入力 u(=input value, etc voltage, motor torque ...)
CMatrix v; // 追加入力雑音 v 入力数と同じ数のベクトル、各雑音は独立して生成
CMatrix xi; // 識別信号 ξ
CMatrix C; // 状態変数の重み行列 C
CMatrix D; // 入力の重み行列 D
CMatrix H; // 不確実な動作に対する統計処理結果を格納する行列 H
CMatrix P; // リカッチ方程式の解
CMatrix Gamma; // 感度計算に用いる行列 Γ
CMatrix Gamma_aa; // 感度計算に用いる行列 Γaa
CMatrix Gamma_ab; // 感度計算に用いる行列 Γab
CMatrix Gamma_ba; // 感度計算に用いる行列 Γba
CMatrix Gamma_bb; // 感度計算に用いる行列 Γbb
CMatrix Psi; // 感度計算に用いる行列 Ψ
CMatrix Upsilon; // 感度計算に用いる行列 Υ
CMatrix Xi; // 感度計算に用いる行列 Ξ
CMatrix G; // ゲイン G(=Feedback gain, etc k ...)

CMatrix I; // 単位行列I

// 後で切り離すか考える
CMatrix A;
CMatrix B;

// ---------------
// 共通するモノ
// ---------------
dWorldID world;  // 動力学の世界
dSpaceID space;  // 衝突検出用スペース
dGeomID  ground; // 地面
dJointGroupID contactgroup;  // ジョイントグループ(物体と地面との衝突に使用)
dsFunctions fn;  // ODEに用意されている関数のエントリを格納

// -----------------
// ジョイントの定義(回転軸に使用)
// -----------------
/*
#define RotalyPendulumJointMax 3
dJointID RotalyPendulumJoint[RotalyPendulumJointMax];
*/

#define StandCarPendulumJointMax 2 // 3
dJointID StandCarPendulumJoint[StandCarPendulumJointMax];

// -------------
// 物体の定義
// -------------
/*
MyObject FixedStand; // アームの固定台
MyObject DriveArm; // アーム
MyObject Pendulum1; // 1番目の振り子
*/

MyObject Wheel1; // 車輪1
MyObject Wheel2; // 車輪2
MyObject Stand; // 台車
MyObject Pendulum1; // 振り子

#define WHEEL1_MASS 1.0 // 車輪1の重さ
#define WHEEL1_RADIUS 0.2 // 車輪1の半径
#define WHEEL1_LENGTH 0.3 // 車輪1の厚み

#define WHEEL2_MASS WHEEL1_MASS // 車輪2の重さ
#define WHEEL2_RADIUS WHEEL1_RADIUS // 車輪2の半径
#define WHEEL2_LENGTH WHEEL1_LENGTH // 車輪2の厚み

#define STAND_MASS 1.0 // 台車の台の重さ
#define STAND_SIZE_X 3.0 // 台車のサイズ
#define STAND_SIZE_Y 0.18 // 台車のサイズ
#define STAND_SIZE_Z 0.2 // 台車のサイズ
#define STAND_START_Z WHEEL1_RADIUS// 位置

#define PENDULUM1_MASS 0.3 // 振り子1の重さ
#define PENDULUM1_SIZE_X 0.015 // 
#define PENDULUM1_SIZE_Y 0.015 //
#define PENDULUM1_SIZE_Z 1.0 //
#define PENDULUM1_START_Z WHEEL1_RADIUS //

/*
#define FIXED_STAND_MASS 1.0 // 固定台の質量 1.0[kg]
#define FIXED_STAND_SIZE_X 0.14 // 固定台のサイズ x 0.12[m]=120[mm]
#define FIXED_STAND_SIZE_Y 0.14 // 固定台のサイズ y 0.12[m]=120[mm]
#define FIXED_STAND_SIZE_Z 0.14 // 固定台のサイズ z 0.12[m]=120[mm]
#define FIXED_STAND_START_Z ((FIXED_STAND_SIZE_Z) / 2.0) // 固定台の初期値

#define DRIVE_ARM_MASS 0.05 // アームの質量 1.0[kg]
#define DRIVE_ARM_SIZE_X 0.19 // 180[mm] アーム長さ
#define DRIVE_ARM_SIZE_Y 0.035 // 39[mm] アームの幅
#define DRIVE_ARM_SIZE_Z 0.030 //  5[mm] アームの厚み
#define DRIVE_ARM_START_X ((DRIVE_ARM_SIZE_X) / 2.0)
#define DRIVE_ARM_START_Z  ((FIXED_STAND_SIZE_Z) + ((DRIVE_ARM_SIZE_Z) / 2.0))

#define PENDULUM1_MASS 0.1
#define PENDULUM1_SIZE_X 0.015
#define PENDULUM1_SIZE_Y 0.015
#define PENDULUM1_SIZE_Z 0.5
#define PENDULUM1_START_X (DRIVE_ARM_SIZE_X)
#define PENDULUM1_START_Z ((FIXED_STAND_SIZE_Z) + ((DRIVE_ARM_SIZE_Z) / 2.0) + (PENDULUM1_SIZE_Z) / 2.0)
*/

void simLoop(int pause); // ODEメイン
void start(); // ODE開始時に呼ばれる関数
void setDrawStuff(); // 描画を受け持つ関数
void nearCallback(void *data, dGeomID o1, dGeomID o2); // 衝突検出用コールバック関数
void stop(); // ODE停止時に呼ばれる関数
void command(); // ODEキーイベント時に呼ばれる関数
void restart(); // ODE再スタート時に呼ばれる関数
void InitALL(); // ODEの初期化関数

//------------------------------------------------------
// 関数の広域宣言(どのc/cppコードからも呼び出し可能)
//------------------------------------------------------
extern void simLoop(int pause); // メインシミュレーションループ
extern void start(); // プログラム開始時に呼ばれる関数
extern void setDrawStuff(); // 描画を受け持つ関数
extern void nearCallback(void *data, dGeomID o1, dGeomID o2); // 衝突検出用コールバック関数
extern void stop(); // ODE停止時に呼ばれる関数
extern void command(); // ODEキーイベント時に呼ばれる関数
extern void restart(); // ODE再スタート時に呼ばれる関数
extern void InitALL(); // ODEの初期化関数
extern void GetRollPitchYaw(dBodyID tmpbody, dReal *rpy, const dReal *angle_rpy); // 指定された物体のロー角、ピッチ角、ヨー角およびその角速度を取得する関数
extern void dJointGetHingeAngleInfiniteDiff(dJointID jointID, double *Angle, double dt); // dJointGetHingeAngleの-∞<-->+∞版

/*
extern void CreateFixedStand();
extern void CreateDriveArm();
extern void CreatePendulum1();
extern void CreateJoint0();
extern void CreateJoint1();
extern void CreateJoint2();
*/

extern void CreateWheel1();
extern void CreateWheel2();
extern void CreateStand();
extern void CreatePendulum1();
extern void CreateJoint0();
extern void CreateJoint1();
extern void CreateJoint2();

using namespace std; // std名前空間を省略

//
// プロットデータファイル
//
ofstream state_file ("./data/state.dat"); // 状態データ
ofstream state_csv_file ("./data/state.csv"); // 状態データ
ofstream gain_file ("./data/gain.dat"); // ゲインデータ
ofstream gain_csv_file ("./data/gain.csv"); // 状態データ
ofstream input_file ("./data/input.dat"); // 入力データ
ofstream input_csv_file ("./data/input.csv"); // 入力データ

void PrintMatrix(const char *name, const CMatrix &m); // 指定された行列のコンソールへの表示
void MatrixTest(void); // 行列の数値テスト

static struct dStopwatch mytimer; // ODE内のタイマー
double realtime = 0.0; // 実時間
double phystime = 0.0; // 物理演算空間内時間
double drawtime = 0.0; // 描画ウェイトのための時間

void GammaUpdateFromGamma_aa_ab_ba_bb(void); // ΓからΓaa、Γab、Γba、Γbbを得る
void Gamma_aa_ab_ba_bbUpdateFromGamma(void); // Γaa、Γab、Γba、ΓbbからΓを得る
void zUpdate(void); // 応答信号zの更新
void xiUpdate(void); // 識別信号ξの更新
void RecordGraphx(void); // 状態のグラフ出力
void RecordGraphu(void); // 入力のグラフ出力
void RecordGraphG(void); // ゲインのグラフ出力
void GainUpdate(void); // ゲインに関わる更新
void SensitivityMain(void); // 感度計算
void Control(void); // 制御入力を決定
void ResetModel(void); // 制御対象のモデルをリセット
void InitializeAllParameter(void); // 理論の全パラメータを初期化

double Wheel1Angle;
double Wheel2Angle;

#endif